/*
 * ________________________________________________________________________________________________________
 * Copyright (c) 2020 GebraBit Inc. All rights reserved.
 *
 * This software, related documentation and any modifications thereto (collectively Software) is subject
 * to GebraBit and its licensors' intellectual property rights under U.S. and international copyright
 * and other intellectual property rights laws. 
 *
 * GebraBit and its licensors retain all intellectual property and proprietary rights in and to the Software
 * and any use, reproduction, disclosure or distribution of the Software without an express license agreement
 * from GebraBit is strictly prohibited.
 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT 
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT IN  
 * NO EVENT SHALL GebraBit BE LIABLE FOR ANY DIRECT, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, 
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THE SOFTWARE.
 * @Author       	: Mehrdad Zeinali
 * ________________________________________________________________________________________________________
 */
#include "GebraBit_MS5637.h"

extern I2C_HandleTypeDef 		hi2c1;

/*========================================================================================================================================= 
 * @brief     Read multiple data from first spacial register address.
 * @param     regAddr First Register Address of MS5611 that reading multiple data start from this address
 * @param     data    Pointer to Variable that multiple data is saved .
 * @param     byteQuantity Quantity of data that we want to read .
 * @return    None
 ========================================================================================================================================*/
void GB_MS5637_Burst_Read(uint8_t regAddr,  uint8_t *data, uint16_t byteQuantity)																			/*		Read Burst Data From Register			*/
{
	HAL_I2C_Mem_Read(MS5637_I2C,MS5637_READ_ADD,regAddr,1,data,byteQuantity,200);
}
/*=========================================================================================================================================
 * @param     cmd    Command that will be writen 
 * @return    status    Return status
 ========================================================================================================================================*/
void GB_MS5637_Write_Command( uint8_t cmd)
{
	uint8_t TBuff[1];
	TBuff[0]=cmd;
	HAL_I2C_Master_Transmit(MS5637_I2C,MS5637_WRITE_ADD,TBuff,1,100);
}
/*=========================================================================================================================================
 * @brief     Reset MS5637
 * @param     MS5637   MS5637 Struct
 * @return    Nothing
 ========================================================================================================================================*/
void GB_MS5637_Soft_Reset ( GebraBit_MS5637 * MS5637 )  
{
	GB_MS5637_Write_Command(MS5637_RESET);
	HAL_Delay(5);
}
/*=========================================================================================================================================
 * @brief     Read MS5637 PROM
 * @param     MS5637   MS5637 Struct PROM_DATA variable
 * @return    Nothing
 ========================================================================================================================================*/
void GB_MS5637_Read_PROM ( GebraBit_MS5637 * MS5637 ) 
{
	uint8_t i ;
	for(i=0;i<13;)
  {
	 GB_MS5637_Burst_Read( MS5637_PROM_READ|i,&MS5637->PROM_DATA[i],2); 
	 i+=2;
	}
}

/*=========================================================================================================================================
 * @brief     Read Calibration data (Factory Calibrated Data)
 * @param     MS5637   MS5637 Struct RESET  variable
	C1:Pressure sensitivity | SENST1
	C2:Pressure offset | OFFT1
	C3:Temperature coefficient of pressure sensitivity | TCS
	C4:Temperature coefficient of pressure offset | TCO
	C5:Reference temperature | TREF
	C6:Temperature coefficient of the temperature | TEMPSENS
	CRC_SERIAL_CODE:CRC to check the data validity in memory
 * @return    Nothing
 ========================================================================================================================================*/
void GB_MS5637_Read_Factory_Calibrated_Data ( GebraBit_MS5637 * MS5637 ) 
{
	GB_MS5637_Read_PROM ( MS5637 );
	MS5637->CRC_SERIAL_CODE = (((uint16_t)(((uint16_t)(MS5637->PROM_DATA[0] << 8))| MS5637->PROM_DATA[1]))& 0xF000) >> 12;
	MS5637->C1 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[2] << 8))| MS5637->PROM_DATA[3]);
	MS5637->C2 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[4] << 8))| MS5637->PROM_DATA[5]);
	MS5637->C3 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[6] << 8))| MS5637->PROM_DATA[7]);
	MS5637->C4 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[8] << 8))| MS5637->PROM_DATA[9]);
	MS5637->C5 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[10] << 8))| MS5637->PROM_DATA[11]);
	MS5637->C6 = (uint16_t)(((uint16_t)(MS5637->PROM_DATA[12] << 8))| MS5637->PROM_DATA[13]);	
}
/*=========================================================================================================================================
 * @brief     Set Pressure Output Sample Rate
 * @param     MS5637     MS5637 Struct PRESSURE_SAMPLE_RATE variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_Pressure_Sample_Rate(GebraBit_MS5637 * MS5637 , MS5637_Output_Sample_Rate rate)
{
	MS5637->PRESSURE_SAMPLE_RATE = rate ;
}	
/*=========================================================================================================================================
 * @brief     Set Temperature Output Sample Rate
 * @param     MS5637     MS5637 Struct TEMPERATURE_SAMPLE_RATE variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void	GB_MS5637_Temperature_Sample_Rate(GebraBit_MS5637 * MS5637 , MS5637_Output_Sample_Rate rate)
{
	MS5637->TEMPERATURE_SAMPLE_RATE = rate ;
}
/*=========================================================================================================================================
 * @brief     Start MS5637 ADC to sample Pressure  
 * @param     MS5637     MS5637 Struct 
 * @return    Nothing
 ========================================================================================================================================*/ 
void	GB_MS5637_Start_Pressure_Sampling(GebraBit_MS5637 * MS5637)
{	
	GB_MS5637_Write_Command(MS5637_PRESSURE_SAMPLING_START|MS5637->PRESSURE_SAMPLE_RATE);
	switch(MS5637->PRESSURE_SAMPLE_RATE)
	 {
	  case OSR_256:
		HAL_Delay(MS5637_OSR_256_CONVERSION_TIME);
    break;
		case OSR_512:
		HAL_Delay(MS5637_OSR_512_CONVERSION_TIME);
    break;	
		case OSR_1024:
		HAL_Delay(MS5637_OSR_1024_CONVERSION_TIME);
    break;	
		case OSR_2048:
		HAL_Delay(MS5637_OSR_2048_CONVERSION_TIME);
    break;			
		case OSR_4096:
		HAL_Delay(MS5637_OSR_4096_CONVERSION_TIME);
    break;		
		case OSR_8192:  
		HAL_Delay(MS5637_OSR_8192_CONVERSION_TIME);
    break;
	 }
}	
/*=========================================================================================================================================
 * @brief     Start MS5637 ADC to sample Temperature  
 * @param     MS5637     MS5637 Struct 
 * @return    Nothing
 ========================================================================================================================================*/ 
void	GB_MS5637_Start_Temperature_Sampling(GebraBit_MS5637 * MS5637)
{	
	GB_MS5637_Write_Command(MS5637_TEMPERATURE_SAMPLING_START|MS5637->TEMPERATURE_SAMPLE_RATE);
	switch(MS5637->TEMPERATURE_SAMPLE_RATE)
	 {
	  case OSR_256:
		HAL_Delay(MS5637_OSR_256_CONVERSION_TIME);
    break;
		case OSR_512:
		HAL_Delay(MS5637_OSR_512_CONVERSION_TIME);
    break;	
		case OSR_1024:
		HAL_Delay(MS5637_OSR_1024_CONVERSION_TIME);
    break;	
		case OSR_2048:
		HAL_Delay(MS5637_OSR_2048_CONVERSION_TIME);
    break;			
		case OSR_4096:
		HAL_Delay(MS5637_OSR_4096_CONVERSION_TIME);
    break;		
		case OSR_8192:  
		HAL_Delay(MS5637_OSR_8192_CONVERSION_TIME);
    break;
	 }
}	
/*=========================================================================================================================================
 * @brief     initialize MS5637
 * @param     MS5637     MS5637 Struct 
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_initialize( GebraBit_MS5637 * MS5637 )
{
  GB_MS5637_Soft_Reset ( MS5637 ) ;
	GB_MS5637_Read_Factory_Calibrated_Data( MS5637 ) ;
	GB_MS5637_Pressure_Sample_Rate(MS5637 , OSR_4096);
	GB_MS5637_Temperature_Sample_Rate(MS5637 , OSR_4096);
}
/*=========================================================================================================================================
 * @brief     Read MS5637 Raw ADC data
 * @param     MS5637   MS5637 Struct ADC_DATA variable
 * @return    Nothing
 ========================================================================================================================================*/
void GB_MS5637_Read_ADC ( GebraBit_MS5637 * MS5637  ) 
{ 
	GB_MS5637_Burst_Read( MS5637_ADC_READ,MS5637->ADC_DATA,ADC_DATA_BUFFER_SIZE);
}
/*=========================================================================================================================================
 * @brief     Read MS5637 Raw Pressure ADC data
 * @param     MS5637     MS5637 Struct ADC_RAW_PRESSURE variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_Read_ADC_Raw_Pressure(GebraBit_MS5637* MS5637)
{
  GB_MS5637_Start_Pressure_Sampling( MS5637 );
	GB_MS5637_Read_ADC (MS5637) ;
	MS5637->ADC_RAW_PRESSURE =(uint32_t)(((uint32_t)(MS5637->ADC_DATA[0] << 16)) | ((uint32_t)(MS5637->ADC_DATA[1] << 8))| MS5637->ADC_DATA[2]);
} 
/*=========================================================================================================================================
 * @brief     Read MS5637 Raw Temperature ADC data
 * @param     MS5637     MS5637 Struct ADC_RAW_TEMPERATURE variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_Read_ADC_Raw_Temperature(GebraBit_MS5637* MS5637)
{
  GB_MS5637_Start_Temperature_Sampling( MS5637);
	GB_MS5637_Read_ADC (MS5637) ;
	MS5637->ADC_RAW_TEMPERATURE =(uint32_t)(((uint32_t)(MS5637->ADC_DATA[0] << 16)) | ((uint32_t)(MS5637->ADC_DATA[1] << 8))| MS5637->ADC_DATA[2]);
}
/*=========================================================================================================================================
 * @brief     Calculate MS5637 Temperature According to calibration coefficients
 * @param     MS5637     MS5637 Struct calibration coefficients variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_Calculate_Temperature(GebraBit_MS5637* MS5637)
{
	int32_t TEMP;
  GB_MS5637_Read_ADC_Raw_Temperature(MS5637);
	MS5637->DT=(int32_t)MS5637->ADC_RAW_TEMPERATURE-((int32_t)MS5637->C5<<8);
	TEMP = 2000+((int64_t)MS5637->DT*(int64_t)MS5637->C6>>23) ;
	// Second order temperature compensation
	if (TEMP<2000)
	{
	  MS5637->T2=(3*((int64_t)MS5637->DT*(int64_t)MS5637->DT))>>33;
		MS5637->OFF2 = 61 * ((int64_t)TEMP - 2000) * ((int64_t)TEMP - 2000) / 16 ;
	  MS5637->SENS2 = 29 * ((int64_t)TEMP - 2000) * ((int64_t)TEMP - 2000) / 16 ;
	  if( TEMP < -1500 )
		{
			MS5637->OFF2 += 17 * ((int64_t)TEMP + 1500) * ((int64_t)TEMP + 1500) ;
			MS5637->SENS2 += 9 * ((int64_t)TEMP + 1500) * ((int64_t)TEMP + 1500) ;
		}
	}
	else
	{
		MS5637->T2 = ( 5 * ( (int64_t)MS5637->DT  * (int64_t)MS5637->DT  ) ) >> 38;
		MS5637->OFF2 = 0 ;
		MS5637->SENS2 = 0 ;
	}
  MS5637->TEMPERATURE = ( (float)TEMP - MS5637->T2 ) / 100;	
}
/*=========================================================================================================================================
 * @brief     Calculate MS5637 Temperature Compensated Pressure According to calibration coefficients
 * @param     MS5637     MS5637 Struct calibration coefficients variable
 * @return    Nothing
 ========================================================================================================================================*/ 
void GB_MS5637_Calculate_Temperature_Compensated_Pressure(GebraBit_MS5637* MS5637)
{
	int64_t P;
  GB_MS5637_Read_ADC_Raw_Pressure(MS5637);
	// OFF = OFF_T1 + TCO * dT
	MS5637->OFF = ( (int64_t)(MS5637->C2) << 17 ) + ( ( (int64_t)(MS5637->C4) * MS5637->DT ) >> 6 ) ;
	MS5637->OFF -= MS5637->OFF2 ;
	
	// Sensitivity at actual temperature = SENS_T1 + TCS * dT
	MS5637->SENS = ( (int64_t)MS5637->C1 << 16 ) + ( ((int64_t)MS5637->C3 * MS5637->DT) >> 7 ) ;
	MS5637->SENS -= MS5637->SENS2 ;
	
	// Temperature compensated pressure = D1 * SENS - OFF
	P = ( ( (MS5637->ADC_RAW_PRESSURE * MS5637->SENS) >> 21 ) - MS5637->OFF ) >> 15 ;
  MS5637->PRESSURE = (float)P/ 100;
}
/*=========================================================================================================================================
 * @brief     Convert Pressuer To Altitude
 * @param     MS5637   MS5637 struct ALTITUDE Variable
 * @return    Nothing
 ========================================================================================================================================*/ 
/*
M403Z 
*/
void GB_MS5637_Altitude(GebraBit_MS5637 * MS5637)
{
    double altitude;

    MS5637->ALTITUDE = ((1 - pow((MS5637->PRESSURE*100) / SEA_LEVEL_PRESSURE, 1/5.257)) / 0.0000225577);  
}

/*----------------------------------------------------------------------------------------------------------------------------------------*
 *                                                                      End                                                               *
 *----------------------------------------------------------------------------------------------------------------------------------------*/